﻿using CSharpCaluc;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Numerics;
using System.Text;
using System.Threading.Tasks;

namespace graphicbox2d
{
    /// <summary>
    /// グラフ図形クラス
    /// </summary>
    public class Graph2D : Object2D, ILineProperty
    {
        // ===============================================================================
        // 公開プロパティ
        // ===============================================================================

        /// <summary>
        /// 図形の種類
        /// </summary>
        public override eObject2DType m_Type => eObject2DType.Graph;

        /// <summary>
        /// グラフ化する数式を設定
        /// ※数式を設定した後、必ず CalculateGraphPoints() メソッドを実行して点リストを計算してください。
        /// 
        /// ＊概要＊
        /// グラフ化する数式を文字列で指定します。
        /// 例：x^(2√2) + 3*x + 2 + |sin(x)÷2|
        /// 
        /// ＊変数名＊
        /// 数式の変数名は「x」のみ使用可能です。
        /// 〇例：sin(x)、x^2 + 3×x +2
        /// ×例：sin(a)、a^2 + 3×a +2
        /// 
        /// ＊使用可能演算子＊
        /// 以下の演算子が使用可能です。
        /// ・掛け算：×、＊
        /// ・割り算：÷、/
        /// ・足し算：＋、+
        /// ・引き算：－、-
        /// ・乗算　：＾、^
        /// 
        /// ＊使用可能記号＊
        /// ・根号　　　：√  （例：√x、√(x+1)）
        /// ・絶対値記号：｜　（例：|x+1|）
        /// 
        /// ＊使用可能定数記号＊
        /// ・円周率　　：π
        /// ・ネイピア数：e
        /// 
        /// ＊使用可能な関数＊
        /// ・アークサイン              ：Asin
        /// ・アークサイン              ：ArcSin
        /// ・アークコサイン            ：Acos
        /// ・アークコサイン            ：ArcCos
        /// ・アークタンジェント        ：Atan
        /// ・アークタンジェント        ：ArcTan
        /// ・サイン                    ：Sin
        /// ・コサイン                  ：Cos
        /// ・タンジェント              ：Tan
        /// ・ハイパボリックサイン      ：Sinh
        /// ・ハイパボリックコサイン    ：Cosh
        /// ・ハイパボリックタンジェント：Tanh
        /// ・対数（自然対数）          ：Log
        /// ・指数関数                  ：Exp        
        /// 
        /// </summary>
        public string Susiki {get;set;} = "sin(x)";

        /// <summary>
        /// グラフ化するXの開始値
        /// </summary>
        public float StartX { get; set; } = -10.0f;

        /// <summary>
        /// グラフ化するXの終了値
        /// </summary>
        public float EndX { get; set; } = 10.0f;

        /// <summary>
        /// 計算ステップ間隔
        /// この値の感覚でグラフの滑らかさが変わります。
        /// 小さくするほど滑らかになりますが、計算に時間がかかります。
        /// </summary>
        public float CalculateInterval { get; set; } = 0.05f;

        /// <summary>
        /// グラフの点リスト
        /// </summary>
        public List<PointF> Points { get; set; } = new List<PointF>();

        /// <summary>
        /// 線の太さ
        /// </summary>
        public int Width { get; set; } = 1;

        /// <summary>
        /// 線の種類
        /// </summary>
        public DashStyle LineStyle { get; set; } = DashStyle.Solid;

        // ===============================================================================
        // 非公開プロパティ
        // ===============================================================================

        /// <summary>
        /// 図形の中心点
        /// </summary>
        internal override Vector2 CenterPoint { get { return GetCenterPoint(); } }

        /// <summary>
        /// 原点とXの値が最も近い点のインデックス値
        /// </summary>
        private int ZeroXPointIndex = -1;

        // ===============================================================================
        // 公開メソッド
        // ===============================================================================

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Graph2D() 
        {
        }

        /// <summary>
        /// グラフの点リストを計算する
        /// ※呼び出し元で待機は不要です。非同期で実行されます。
        /// </summary>
        public async Task CalculateGraphPoints()
        {
            // 計算開始メッセージを送信
            WinAPI.PostMessage(Graphic2DControl.hWnd, WinMsg.WM_SUSIKI_CALC_START, (IntPtr)_ID, IntPtr.Zero);


            // 重い計算はバックグラウンドで実行
            PointF[] points = await Task.Run(() =>
                GraphicCaluculate.SusikiCaluculate(Susiki, StartX, EndX, CalculateInterval, _ID)
            );

            Points = points.ToList();
            ZeroXPointIndex = GetZeroXPointIndex(Points);

            // 計算終了メッセージを送信
            WinAPI.PostMessage(Graphic2DControl.hWnd, WinMsg.WM_SUSIKI_CALC_END, (IntPtr)_ID, IntPtr.Zero);
        }

        /// <summary>
        /// コピーを作成する
        /// </summary>
        /// <returns>コピーしたオブジェクト</returns>
        public override Object2D Clone()
        {
            Graph2D clone = new Graph2D();

            // 基底クラスのデータをコピー
            this.BaseCopyDataTo(clone);

            // 自クラスのデータをコピー
            clone.Susiki = this.Susiki;
            clone.StartX = this.StartX;
            clone.EndX = this.EndX;
            clone.CalculateInterval = this.CalculateInterval;
            clone.Points = new List<PointF>(this.Points);
            clone.Width = this.Width;

            return clone;
        }

        // ===============================================================================
        // 非公開メソッド
        // ===============================================================================

        /// <summary>
        /// マウスヒット中の図形（拡大した図形）を返す。
        /// </summary>
        /// <returns>拡張された図形</returns>
        internal override Object2D GetHitObject()
        {
            Graph2D graph2D = (Graph2D)this.Clone();

            graph2D.Width += MouseHitLineOffset;

            return graph2D;
        }

        /// <summary>
        /// マウスポイントがこの図形にヒットしているか判定する。
        /// </summary>
        /// <param name="MousePoint">マウスポイント</param>
        /// <param name="MusePointRange">マウスヒット半径</param>
        /// <returns></returns>
        internal override eMouseHitType IsHitMousePoint(PointF MousePoint, float MusePointRange)
        {
            eMouseHitType eMouseHitType;

            eMouseHitType = GraphicCaluculate.IsHitMouseRangeLineGraph(Points, Width, MousePoint, MusePointRange);

            return eMouseHitType;
        }

        /// <summary>
        /// マウスポイントとこの図形の距離を取得する
        /// </summary>
        /// <param name="X">マウスポイントX座標</param>
        /// <param name="Y">マウスポイントY座標</param>
        /// <returns>距離</returns>
        internal override float GetDistanceHitMousePoint(float X, float Y)
        {
            Vector2 MousePoint = new Vector2(X, Y);

            return Vector2.Distance(MousePoint, CenterPoint);
        }

        /// <summary>
        /// 図形を移動させる
        /// </summary>
        /// <param name="Movement">移動量</param>
        internal override void Move(PointF Movement)
        {
            for (int i = 0; i < Points.Count; i++)
            {
                Points[i] = new PointF(Points[i].X + Movement.X, Points[i].Y + Movement.Y);
            }
        }

        /// <summary>
        /// 図形を移動させる
        /// </summary>
        /// <param name="X">移動量X</param>
        /// <param name="Y">移動量Y</param>
        internal override void Move(float X, float Y)
        {
            for (int i = 0; i < Points.Count; i++)
            {
                Points[i] = new PointF(Points[i].X + X, Points[i].Y + Y);
            }
        }

        /// <summary>
        /// バウンディングボックスを取得する
        /// </summary>
        internal override PointF[] GetBoundingBox()
        {
            return GraphicCaluculate.GetBoundingBoxPolygon(Points.ToArray());
        }

        /// <summary>
        /// 中心点を取得する
        /// </summary>
        /// <returns></returns>
        internal Vector2 GetCenterPoint()
        {
            if (ZeroXPointIndex == -1 || Points == null || Points.Count == 0)
            {
                return new Vector2(0, 0);
            }

            return Points[ZeroXPointIndex].ToVector2();
        }

        /// <summary>
        /// 引数の点リストからXの値が0に最も近い点のインデックス値を取得する
        /// </summary>
        /// <param name="Points">点リスト</param>
        /// <returns>インデックス値。リストが空なら -1 を返す</returns>
        private int GetZeroXPointIndex(List<PointF> Points)
        {
            if (Points == null || Points.Count == 0)
            {
                return -1; // 空リストなら -1
            }

            int closestIndex = 0;
            float minDistance = Math.Abs(Points[0].X);

            for (int i = 1; i < Points.Count; i++)
            {
                float distance = Math.Abs(Points[i].X);
                if (distance < minDistance)
                {
                    minDistance = distance;
                    closestIndex = i;
                }
            }

            return closestIndex;
        }
    }
}
